
// === Mapa de errores por código PixelPay (response_code) ===
// Amplíalo si tu comercio maneja códigos extra.
const PIXEL_ERRORS = {
	// Genéricos
	"01": "Transacción declinada por el emisor.",
	"02": "Banco emisor no disponible.",
	"03": "Comercio inválido o no configurado.",
	"05": "Transacción no aprobada.",
	"12": "Transacción inválida.",
	"13": "Monto inválido o fuera de rango.",
	"14": "Tarjeta inválida.",
	"15": "Emisor no encontrado.",
	"51": "Fondos insuficientes.",
	"54": "Tarjeta expirada.",
	"57": "Transacción no permitida.",
	"58": "Comercio no permite este tipo de operación.",
	"91": "Banco emisor no disponible.",
	"96": "Error general del sistema.",
	// Errores de autenticación/llaves comunes
	"KEY_INVALID": "El valor de la KEY/HASH es inválido. Verifica tus credenciales.",
	"UNAUTHORIZED": "No autorizado: revisa x-auth-key y x-auth-hash.",
};

// === Casos de SANDBOX por amount (documentación de PixelPay) ===
// Útil cuando uses sandbox y quieras forzar outcomes por monto.
const SANDBOX_CASES = {
	1: { title: "Transacción Exitosa", icon: "success" },
	2: { title: "Transacción Declinada", icon: "error" },
	3: { title: "Configuración de comercio inválida", icon: "error" },
	4: { title: "Tarjeta con reporte de robo o extravío", icon: "error" },
	5: { title: "Error al encontrar cobro", icon: "error" },
	6: { title: "Se superó el límite de intentos", icon: "error" },
	7: { title: "Error general del sistema", icon: "error" },
	8: { title: "Tiempo de espera excedido (timeout)", icon: "warning" },
	9: { title: "Monto de transacción excedido", icon: "error" },
	10: { title: "Límite de transacciones excedido", icon: "error" },
	11: { title: "Límite por monto de transacciones excedido", icon: "error" },
	12: { title: "Límite por número de BIN", icon: "error" },
	13: { title: "Límite por tipo de tarjeta", icon: "error" },
	14: { title: "Límite por localización", icon: "error" },
};

// Detecta si estamos en sandbox por variable pública del widget (opcional).
// Si no tienes esa variable en window, elimina este helper y usa tu propia bandera.
function isSandbox() {
	// ajusta si expones tu ENV al front: window.BMT_ENV o similar
	try { return (window.BMT_ENV || "sandbox") === "sandbox"; } catch { return true; }
}

// Mensaje final para SweetAlert a partir del resultado/errores
function buildFrontMessage({ pixelMsg, pixelCodes, result, amount }) {
	// 1) Si hay response_code conocido → usa texto curado
	if (pixelCodes?.code && PIXEL_ERRORS[pixelCodes.code]) {
		return {
			title: "Pago no aprobado",
			icon: "error",
			html: `
          <div style="text-align:left">
            <p><b>Detalle:</b> ${PIXEL_ERRORS[pixelCodes.code]}</p>
            ${pixelMsg ? `<p><b>Mensaje:</b> ${pixelMsg}</p>` : ""}
            <p><b>Código:</b> ${pixelCodes.code}</p>
            ${pixelCodes.uuid ? `<p><b>UUID:</b> ${pixelCodes.uuid}</p>` : ""}
          </div>`
		};
	}

	// 2) Si sandbox y el monto está en la tabla → usa el caso sandbox
	if (isSandbox() && Number.isFinite(amount) && SANDBOX_CASES[amount]) {
		const c = SANDBOX_CASES[amount];
		return {
			title: c.title,
			icon: c.icon,
			html: `
          <div style="text-align:left">
            ${pixelMsg ? `<p><b>Mensaje:</b> ${pixelMsg}</p>` : `<p>Simulación sandbox con amount = ${amount}</p>`}
            ${pixelCodes?.code ? `<p><b>Código:</b> ${pixelCodes.code}</p>` : ""}
          </div>`
		};
	}

	// 3) Si PixelPay devolvió algo como KEY inválida / Unauthorized en texto libre
	const rawText = (result?.message || pixelMsg || "").toLowerCase();
	if (rawText.includes("unauthorized") || rawText.includes("key") || rawText.includes("hash")) {
		return {
			title: "Credenciales PixelPay",
			icon: "error",
			html: `
          <div style="text-align:left">
            <p><b>Detalle:</b> El valor de la KEY/HASH es inválido o no autorizado.</p>
            ${pixelMsg ? `<p><b>Mensaje:</b> ${pixelMsg}</p>` : ""}
            ${pixelCodes?.code ? `<p><b>Código:</b> ${pixelCodes.code}</p>` : ""}
          </div>`
		};
	}

	// 4) Fallback genérico
	return {
		title: "Pago no aprobado",
		icon: "error",
		html: `
        <div style="text-align:left">
          ${pixelMsg ? `<p><b>Mensaje:</b> ${pixelMsg}</p>` : "<p>No se pudo procesar la transacción.</p>"}
          ${pixelCodes?.code ? `<p><b>Código:</b> ${pixelCodes.code}</p>` : ""}
        </div>`
	};
}

