<?php
/* public/guardar-usuario.php */
declare(strict_types=1);
require __DIR__ . '/config.php';
header('Content-Type: application/json; charset=utf-8');

try {
  // === Leer datos tanto de JSON como de form-data ===
  // Si ya tienes request_data() en config.php úsalo; si no, este fallback:
  if (!function_exists('request_data')) {
    function request_data(): array {
      $raw  = file_get_contents('php://input');
      $json = json_decode($raw, true);
      return (is_array($json) && !empty($json)) ? $json : $_POST;
    }
  }

  $src = request_data();

  // === Helpers de saneo ===
  $str = static function($v): string { return trim((string)$v); };
  $up  = static function($v): string { return strtoupper(trim((string)$v)); };

  // Email simple
  $email = $str($src['email'] ?? '');
  if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    throw new RuntimeException('Email inválido');
  }

  // Teléfono: quitar espacios, puntos, guiones
  $phone = preg_replace('/[\s\.\-]/', '', (string)($src['billing']['phone'] ?? $src['phone'] ?? ''));
  $phone = $str($phone);

  // Cargar usuario existente para preservar campos (balance, id, etc.)
  $current = [];
  if (defined('USER_JSON_PATH') && file_exists(USER_JSON_PATH)) {
    $raw = file_get_contents(USER_JSON_PATH);
    $cur = json_decode($raw, true);
    if (is_array($cur)) $current = $cur;
  }

  // Defaults (por si no existían)
  $defId      = $current['id']       ?? 'u_demo_001';
  $defBalance = (float)($current['balance'] ?? 0);
  $defCurr    = $current['currency'] ?? 'HNL';

  // Construir nuevo objeto usuario (mergeando lo previo)
  $u = [
    'id'       => $defId,
    'name'     => $str($src['name'] ?? ($current['name'] ?? '')),
    'email'    => $email !== '' ? $email : ($current['email'] ?? ''),
    'balance'  => $defBalance,
    'currency' => $up($src['currency'] ?? $defCurr),
    'billing'  => [
      'address' => $str($src['billing']['address'] ?? $src['address'] ?? ($current['billing']['address'] ?? '')),
      'country' => $up($src['billing']['country'] ?? $src['country'] ?? ($current['billing']['country'] ?? 'HN')),
      'state'   => $up($src['billing']['state']   ?? $src['state']   ?? ($current['billing']['state']   ?? '')),
      'city'    => $str($src['billing']['city']   ?? $src['city']    ?? ($current['billing']['city']    ?? '')),
      'phone'   => $phone !== '' ? $phone : ($current['billing']['phone'] ?? ''),
    ],
    'session' => [
      'logged_in'  => true,
      // si ya existe created_at lo respetamos
      'created_at' => $current['session']['created_at'] ?? date('c'),
      'updated_at' => date('c'),
    ],
  ];

  // Reglas mínimas
  if ($u['name'] === '')   throw new RuntimeException('El nombre es requerido');
  if ($u['email'] === '')  throw new RuntimeException('El email es requerido');
  if ($u['billing']['country'] === '') $u['billing']['country'] = 'HN';

  // === Guardar atómicamente ===
  $json = json_encode($u, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
  if ($json === false) throw new RuntimeException('No se pudo serializar JSON');

  $ok = false;
  $fp = fopen(USER_JSON_PATH, 'c+');
  if ($fp === false) throw new RuntimeException('No se pudo abrir el archivo de usuario');

  try {
    if (!flock($fp, LOCK_EX)) throw new RuntimeException('No se pudo bloquear el archivo');
    ftruncate($fp, 0);
    rewind($fp);
    $written = fwrite($fp, $json);
    fflush($fp);
    flock($fp, LOCK_UN);
    fclose($fp);
    $ok = ($written !== false);
  } catch (\Throwable $e) {
    try { flock($fp, LOCK_UN); fclose($fp); } catch (\Throwable $e2) {}
    throw $e;
  }

  if (!$ok) throw new RuntimeException('No se pudo escribir user.json');

  echo json_encode(['success' => true, 'user' => $u], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(400);
  echo json_encode(['success' => false, 'message' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
}
